<?php
/* Copyright (C) 2004-2017 Laurent Destailleur  <eldy@users.sourceforge.net>
 * Copyright (C) 2021 Gauthier VERDOL <gauthier.verdol@atm-consulting.fr>
 * Copyright (C) 2021 SuperAdmin
 * Copyright (C) 2024-2025	MDW							<mdeweerd@users.noreply.github.com>
 * Copyright (C) 2024-2025  Frédéric France             <frederic.france@free.fr>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

/**
 * \file    stocktransfer/admin/setup.php
 * \ingroup stocktransfer
 * \brief   StockTransfer setup page.
 */

// Load Dolibarr environment
require '../main.inc.php';

// Libraries
require_once DOL_DOCUMENT_ROOT."/core/lib/admin.lib.php";
require_once DOL_DOCUMENT_ROOT.'/product/stock/stocktransfer/class/stocktransfer.class.php';
require_once DOL_DOCUMENT_ROOT.'/product/stock/stocktransfer/lib/stocktransfer.lib.php';

/**
 * @var Conf $conf
 * @var DoliDB $db
 * @var HookManager $hookmanager
 * @var Translate $langs
 * @var User $user
 */

// Translations
$langs->loadLangs(array("admin", "stocks"));

// Access control
if (!$user->admin) {
	accessforbidden();
}

// Parameters
$action = GETPOST('action', 'alpha');
$backtopage = GETPOST('backtopage', 'alpha');

$value = GETPOST('value', 'alpha');
$label = GETPOST('label', 'alpha');
$scandir = GETPOST('scan_dir', 'alpha');

$arrayofparameters = array(
	'STOCKTRANSFER_MYPARAM1' => array('css' => 'minwidth200', 'enabled' => 1),
	'STOCKTRANSFER_MYPARAM2' => array('css' => 'minwidth500', 'enabled' => 1)
);

$error = 0;


/*
 * Actions
 */

include DOL_DOCUMENT_ROOT.'/core/actions_setmoduleoptions.inc.php';


if ($action == 'updateMask') {
	$maskconststocktransfer = GETPOST('maskconststocktransfer', 'aZ09');
	$maskstocktransfer = GETPOST('maskStockTransfer', 'alpha');

	if ($maskconststocktransfer && preg_match('/_MASK$/', $maskconststocktransfer)) {
		$res = dolibarr_set_const($db, $maskconststocktransfer, $maskstocktransfer, 'chaine', 0, '', $conf->entity);
		if ($res <= 0) {
			$error++;
		}
	}

	if (!$error) {
		setEventMessages($langs->trans("SetupSaved"), null, 'mesgs');
	} else {
		setEventMessages($langs->trans("Error"), null, 'errors');
	}
} elseif ($action == 'specimen') {
	$modele = GETPOST('module', 'alpha');
	$tmpobjectkey = 'StockTransfer';

	$tmpobject = new $tmpobjectkey($db);
	$tmpobject->initAsSpecimen();

	// Search template files
	$file = '';
	$classname = '';
	$dirmodels = array_merge(array('/'), (array) $conf->modules_parts['models']);
	foreach ($dirmodels as $reldir) {
		$file = dol_buildpath($reldir."core/modules/stocktransfer/doc/pdf_".$modele.".modules.php", 0);
		if (file_exists($file)) {
			$classname = "pdf_".$modele;
			break;
		}
	}

	if ($classname !== '') {
		require_once $file;

		$module = new $classname($db);
		'@phan-var-force ModelePDFStockTransfer $module';
		/** @var ModelePDFStockTransfer $module */

		if ($module->write_file($tmpobject, $langs) > 0) {
			header("Location: ".DOL_URL_ROOT."/document.php?modulepart=".strtolower($tmpobjectkey)."&file=SPECIMEN.pdf");
			return;
		} else {
			setEventMessages($module->error, null, 'errors');
			dol_syslog($module->error, LOG_ERR);
		}
	} else {
		setEventMessages($langs->trans("ErrorModuleNotFound"), null, 'errors');
		dol_syslog($langs->trans("ErrorModuleNotFound"), LOG_ERR);
	}
} elseif ($action == 'set') { // Activate a model
	$ret = addDocumentModel($value, 'stocktransfer', $label, $scandir);
} elseif ($action == 'del') {
	$tmpobjectkey = 'StockTransfer';

	$ret = delDocumentModel($value, 'stocktransfer');
	if ($ret > 0) {
		$constforval = strtoupper($tmpobjectkey).'_ADDON_PDF';
		if (getDolGlobalString($constforval) == "$value") {
			dolibarr_del_const($db, $constforval, $conf->entity);
		}
	}
} elseif ($action == 'setdoc') { // Set default model
	$tmpobjectkey = 'StockTransfer';
	$constforval = strtoupper($tmpobjectkey).'_ADDON_PDF';
	if (dolibarr_set_const($db, $constforval, $value, 'chaine', 0, '', $conf->entity)) {
		// The constant that was read before the new set
		// We therefore requires a variable to have a coherent view
		$conf->global->$constforval = $value;
	}

	// On active le modele
	$ret = delDocumentModel($value, 'stocktransfer');
	if ($ret > 0) {
		$ret = addDocumentModel($value, 'stocktransfer', $label, $scandir);
	}
} elseif ($action == 'setmod') {
	// TODO Check if numbering module chosen can be activated
	// by calling method canBeActivated
	$tmpobjectkey = 'StockTransfer';
	$constforval = 'STOCKTRANSFER_'.strtoupper($tmpobjectkey)."_ADDON";
	dolibarr_set_const($db, $constforval, $value, 'chaine', 0, '', $conf->entity);
}



/*
 * View
 */

$form = new Form($db);

$dirmodels = array_merge(array('/'), (array) $conf->modules_parts['models']);

$page_name = "StockTransferSetup";
llxHeader('', $langs->trans($page_name), '', '', 0, 0, '', '', '', 'mod-admin page-stocktransfer');

// Subheader
$linkback = '<a href="'.dolBuildUrl(DOL_URL_ROOT.'/admin/modules.php', ['restore_lastsearch_values' => 1]).'">'.img_picto($langs->trans("BackToModuleList"), 'back', 'class="pictofixedwidth"').'<span class="hideonsmartphone">'.$langs->trans("BackToModuleList").'</span></a>';

print load_fiche_titre($langs->trans($page_name), $linkback, 'stock');

// Configuration header
$head = stocktransferAdminPrepareHead();
print dol_get_fiche_head($head, 'settings', '', -1, "stocktransfer@stocktransfer");

$moduledir = 'stocktransfer';
$myTmpObjects = array();
$myTmpObjects[$moduledir] = array('includerefgeneration' => 1, 'includedocgeneration' => 1, 'class' => 'StockTransfer');

foreach ($myTmpObjects as $myTmpObjectKey => $myTmpObjectArray) {
	// Orders Numbering model
	print load_fiche_titre($langs->trans("NumberingModules", $myTmpObjectKey), '', '');

	print '<div class="div-table-responsive-no-min">'; // You can use div-table-responsive-no-min if you don't need reserved height for your table
	print '<table class="noborder centpercent">';
	print '<tr class="liste_titre">';
	print '<td>'.$langs->trans("Name").'</td>';
	print '<td>'.$langs->trans("Description").'</td>';
	print '<td class="nowrap">'.$langs->trans("Example").'</td>';
	print '<td class="center" width="60">'.$langs->trans("Status").'</td>';
	print '<td class="center" width="16">'.$langs->trans("ShortInfo").'</td>';
	print '</tr>'."\n";

	clearstatcache();

	foreach ($dirmodels as $reldir) {
		$dir = dol_buildpath($reldir."core/modules/".$moduledir);

		if (is_dir($dir)) {
			$handle = opendir($dir);
			if (is_resource($handle)) {
				while (($file = readdir($handle)) !== false) {
					if (strpos($file, 'mod_'.strtolower($myTmpObjectKey).'_') === 0 && substr($file, dol_strlen($file) - 3, 3) == 'php') {
						$file = substr($file, 0, dol_strlen($file) - 4);

						require_once $dir.'/'.$file.'.php';

						$module = new $file($db);

						'@phan-var-force ModeleNumRefStockTransfer $module';

						// Show modules according to features level
						if ($module->version == 'development' && getDolGlobalInt('MAIN_FEATURES_LEVEL') < 2) {
							continue;
						}
						if ($module->version == 'experimental' && getDolGlobalInt('MAIN_FEATURES_LEVEL') < 1) {
							continue;
						}

						if ($module->isEnabled()) {
							dol_include_once('/'.$moduledir.'/class/'.strtolower($myTmpObjectKey).'.class.php');

							print '<tr class="oddeven"><td>'.$module->name."</td><td>\n";
							print $module->info($langs);
							print '</td>';

							// Show example of numbering model
							print '<td class="nowrap">';
							$tmp = $module->getExample();
							if (preg_match('/^Error/', $tmp)) {
								print '<div class="error">'.$langs->trans($tmp).'</div>';
							} elseif ($tmp == 'NotConfigured') {
								print $langs->trans($tmp);
							} else {
								print $tmp;
							}
							print '</td>'."\n";

							print '<td class="center">';
							$constforvar = 'STOCKTRANSFER_'.strtoupper($myTmpObjectKey).'_ADDON';
							if (getDolGlobalString($constforvar) == $file) {
								print img_picto($langs->trans("Activated"), 'switch_on');
							} else {
								print '<a href="'.$_SERVER["PHP_SELF"].'?action=setmod&token='.newToken().'&object='.strtolower($myTmpObjectKey).'&value='.$file.'">';
								print img_picto($langs->trans("Disabled"), 'switch_off');
								print '</a>';
							}
							print '</td>';

							$nameofclass = $myTmpObjectArray['class'];
							$mytmpinstance = new $nameofclass($db);
							$mytmpinstance->initAsSpecimen();

							// Info
							$htmltooltip = '';
							$htmltooltip .= ''.$langs->trans("Version").': <b>'.$module->getVersion().'</b><br>';

							$nextval = $module->getNextValue($mytmpinstance);
							if ("$nextval" != $langs->trans("NotAvailable")) {  // Keep " on nextval
								$htmltooltip .= ''.$langs->trans("NextValue").': ';
								if ($nextval) {
									if (preg_match('/^Error/', $nextval) || $nextval == 'NotConfigured') {
										$nextval = $langs->trans($nextval);
									}
									$htmltooltip .= $nextval.'<br>';
								} else {
									$htmltooltip .= $langs->trans($module->error).'<br>';
								}
							}

							print '<td class="center">';
							print $form->textwithpicto('', $htmltooltip, 1, 'info');
							print '</td>';

							print "</tr>\n";
						}
					}
				}
				closedir($handle);
			}
		}
	}
	print "</table></div><br>\n";

	// Document templates generators
	$type = strtolower($myTmpObjectKey);

	print load_fiche_titre($langs->trans("DocumentModules", $myTmpObjectKey), '', '');

	// Load array def with activated templates
	$def = array();
	$sql = "SELECT nom";
	$sql .= " FROM ".MAIN_DB_PREFIX."document_model";
	$sql .= " WHERE type = '".$db->escape($type)."'";
	$sql .= " AND entity = ".((int) $conf->entity);
	$resql = $db->query($sql);
	if ($resql) {
		$i = 0;
		$num_rows = $db->num_rows($resql);
		while ($i < $num_rows) {
			$array = $db->fetch_array($resql);
			if (is_array($array)) {
				array_push($def, $array[0]);
			}
			$i++;
		}
	} else {
		dol_print_error($db);
	}

	print '<div class="div-table-responsive">'; // You can use div-table-responsive-no-min if you don't need reserved height for your table
	print '<table class="noborder centpercent">'."\n";
	print '<tr class="liste_titre">'."\n";
	print '<td>'.$langs->trans("Name").'</td>';
	print '<td>'.$langs->trans("Description").'</td>';
	print '<td class="center" width="60">'.$langs->trans("Status")."</td>\n";
	print '<td class="center" width="60">'.$langs->trans("Default")."</td>\n";
	print '<td class="center" width="38">'.$langs->trans("ShortInfo").'</td>';
	print '<td class="center" width="38">'.$langs->trans("Preview").'</td>';
	print "</tr>\n";

	clearstatcache();

	foreach ($dirmodels as $reldir) {
		foreach (array('', '/doc') as $valdir) {
			$realpath = $reldir."core/modules/".$moduledir.$valdir;
			$dir = dol_buildpath($realpath);

			if (is_dir($dir)) {
				$handle = opendir($dir);
				if (is_resource($handle)) {
					$filelist = array();
					while (($file = readdir($handle)) !== false) {
						$filelist[] = $file;
					}
					closedir($handle);
					arsort($filelist);

					foreach ($filelist as $file) {
						if (preg_match('/\.modules\.php$/i', $file) && preg_match('/^(pdf_|doc_)/', $file)) {
							if (file_exists($dir.'/'.$file)) {
								$name = substr($file, 4, dol_strlen($file) - 16);
								$classname = substr($file, 0, dol_strlen($file) - 12);

								require_once $dir.'/'.$file;
								$module = new $classname($db);

								'@phan-var-force ModelePDFStockTransfer $module';

								$modulequalified = 1;
								if ($module->version == 'development' && getDolGlobalInt('MAIN_FEATURES_LEVEL') < 2) {
									$modulequalified = 0;
								}
								if ($module->version == 'experimental' && getDolGlobalInt('MAIN_FEATURES_LEVEL') < 1) {
									$modulequalified = 0;
								}

								if ($modulequalified) {
									print '<tr class="oddeven"><td>';
									print(empty($module->name) ? $name : $module->name);
									print "</td><td>\n";
									if (method_exists($module, 'info')) {
										print $module->info($langs);  // @phan-suppress-current-line PhanUndeclaredMethod
									} else {
										print $module->description;
									}
									print '</td>';

									// Active
									if (in_array($name, $def)) {
										print '<td class="center">'."\n";
										print '<a href="'.$_SERVER["PHP_SELF"].'?action=del&value='.urlencode($name).'&token='.newToken().'">';
										print img_picto($langs->trans("Enabled"), 'switch_on');
										print '</a>';
										print '</td>';
									} else {
										print '<td class="center">'."\n";
										print '<a href="'.$_SERVER["PHP_SELF"].'?action=set&value='.urlencode($name).'&scan_dir='.urlencode($module->scandir).'&label='.urlencode($module->name).'&token='.newToken().'">'.img_picto($langs->trans("Disabled"), 'switch_off').'</a>';
										print "</td>";
									}

									// Default
									print '<td class="center">';
									$constforvar = strtoupper($myTmpObjectKey).'_ADDON_PDF';
									if (getDolGlobalString($constforvar) == $name) {
										print img_picto($langs->trans("Default"), 'on');
									} else {
										print '<a href="'.$_SERVER["PHP_SELF"].'?action=setdoc&token='.newToken().'&object='.urlencode($myTmpObjectKey).'&value='.urlencode($name).'&scan_dir='.urlencode($module->scandir).'&label='.urlencode($module->name).'" alt="'.$langs->trans("Default").'">'.img_picto($langs->trans("Disabled"), 'off').'</a>';
									}
									print '</td>';

									// Info
									$htmltooltip = ''.$langs->trans("Name").': '.$module->name;
									$htmltooltip .= '<br>'.$langs->trans("Type").': '.($module->type ? $module->type : $langs->trans("Unknown"));
									if ($module->type == 'pdf') {
										$htmltooltip .= '<br>'.$langs->trans("Width").'/'.$langs->trans("Height").': '.$module->page_largeur.'/'.$module->page_hauteur;
									}
									$htmltooltip .= '<br>'.$langs->trans("Path").': '.preg_replace('/^\//', '', $realpath).'/'.$file;

									$htmltooltip .= '<br><br><u>'.$langs->trans("FeaturesSupported").':</u>';
									$htmltooltip .= '<br>'.$langs->trans("Logo").': '.yn($module->option_logo, 1, 1);
									$htmltooltip .= '<br>'.$langs->trans("MultiLanguage").': '.yn($module->option_multilang, 1, 1);

									print '<td class="center">';
									print $form->textwithpicto('', $htmltooltip, 1, 'info');
									print '</td>';

									// Preview
									print '<td class="center">';
									if ($module->type == 'pdf') {
										print '<a href="'.$_SERVER["PHP_SELF"].'?action=specimen&module='.$name.'&object='.$myTmpObjectKey.'">'.img_object($langs->trans("Preview"), 'generic').'</a>';
									} else {
										print img_object($langs->transnoentitiesnoconv("PreviewNotAvailable"), 'generic');
									}
									print '</td>';

									print "</tr>\n";
								}
							}
						}
					}
				}
			}
		}
	}

	print '</table></div>';
}


print '<br>';
print load_fiche_titre($langs->trans("OtherOptions"), '', '');

print '<div class="div-table-responsive-no-min">'; // You can use div-table-responsive-no-min if you don't need reserved height for your table
print '<table class="noborder centpercent">';

print '<tr class="liste_titre">';
print '<td>'.$langs->trans("Parameter").'</td>';
print '<td></td>';
print '</tr>'."\n";

// Notifications
/*
print '<tr class="oddeven">';
print '<td>'.img_picto('', 'email', 'class="pictofixedwidth"').$langs->trans("Notifications").'</td>';
print '<td colspan="2">';
print $langs->trans("YouMayFindNotificationsFeaturesIntoModuleNotification");
print '</td></tr>';
*/

// More PDF options
print '<tr class="oddeven">';
print '<td>'.img_picto('', 'pdf', 'class="pictofixedwidth"').$langs->trans("MoreOptionsRelatedToPDF").'</td>';
print '<td colspan="2">';
print img_picto('', 'url', 'class="pictofixedwidth"').'<a href="'.DOL_URL_ROOT.'/admin/pdf_other.php">'.$langs->trans("SeeInPDFSetupPage").'</a>';
print '</td></tr>';

print '</table></div>';


// Page end
print dol_get_fiche_end();

llxFooter();
$db->close();
