<?php
/* Copyright (C) 2015   Jean-François Ferry     <jfefe@aternatik.fr>
 * Copyright (C) 2016	Laurent Destailleur		<eldy@users.sourceforge.net>
 * Copyright (C) 2023	Ferran Marcet			<fmarcet@2byte.es>
 * Copyright (C) 2024-2025	MDW					<mdeweerd@users.noreply.github.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

// Create the autoloader for Luracast
require_once DOL_DOCUMENT_ROOT.'/includes/restler/framework/Luracast/Restler/AutoLoader.php';
call_user_func(
	/**
	 * @return Luracast\Restler\AutoLoader
	 */
	static function () {
		$loader = Luracast\Restler\AutoLoader::instance();
		spl_autoload_register($loader);
		return $loader;
	}
);

require_once DOL_DOCUMENT_ROOT.'/includes/restler/framework/Luracast/Restler/iAuthenticate.php';
require_once DOL_DOCUMENT_ROOT.'/includes/restler/framework/Luracast/Restler/iUseAuthentication.php';
require_once DOL_DOCUMENT_ROOT.'/includes/restler/framework/Luracast/Restler/Resources.php';
require_once DOL_DOCUMENT_ROOT.'/includes/restler/framework/Luracast/Restler/Defaults.php';
require_once DOL_DOCUMENT_ROOT.'/includes/restler/framework/Luracast/Restler/RestException.php';

use Luracast\Restler\iAuthenticate;
use Luracast\Restler\Resources;
use Luracast\Restler\Defaults;
use Luracast\Restler\RestException;

/**
 * Dolibarr API access class
 */
class DolibarrApiAccess implements iAuthenticate
{
	const REALM = 'Restricted Dolibarr API';

	/**
	 * @var DoliDB	Database handler
	 */
	public $db;

	/**
	 * @var string[]	Role required by API method		user / external / admin
	 */
	public static $requires = array('user', 'external', 'admin');

	/**
	 * @var string		User role
	 */
	public static $role = 'user';

	/**
	 * @var User		Loggued user
	 */
	public static $user = null;


	/**
	 * Constructor
	 */
	public function __construct()
	{
		global $db;
		$this->db = $db;
	}

	// phpcs:disable PEAR.NamingConventions.ValidFunctionName
	/**
	 * Check access
	 *
	 * @return bool
	 *
	 * @throws RestException 401 Forbidden
	 * @throws RestException 503 Technical error
	 */
	public function __isAllowed()
	{
		// phpcs:enable
		global $conf, $langs, $user;

		$login = '';
		$stored_key = '';

		$userClass = Defaults::$userIdentifierClass;

		/*foreach ($_SERVER as $key => $val)
		{
			dol_syslog($key.' - '.$val);
		}*/

		// api key can be provided in url with parameter api_key=xxx or ni header with header DOLAPIKEY:xxx
		$api_key = '';
		if (isset($_GET['api_key'])) {	// For backward compatibility. Keep $_GET here.
			// TODO Add option to disable use of api key on url. Return errors if used.
			$api_key = $_GET['api_key'];
		}
		if (isset($_GET['DOLAPIKEY'])) {
			// TODO Add option to disable use of api key on url. Return errors if used.
			$api_key = $_GET['DOLAPIKEY']; // With GET method
		}

		// TODO Can filter on user agent.
		//$api_useragent = $_SERVER['HTTP_USER_AGENT'];

		if (isset($_SERVER['HTTP_DOLAPIKEY'])) {        // HTTP Header entry "DOLAPIKEY: ..." can be read with $_SERVER["HTTP_DOLAPIKEY"]
			$api_key = $_SERVER['HTTP_DOLAPIKEY']; 		// With header method (recommended)
		} elseif (empty($api_key)) {
			$headers = getallheaders();					// HTTP Header entry "Authorization: Bearer ..." can be read with getallheaders
			$api_key = preg_replace('/^Bearer\s+/i', '', empty($headers['Authorization']) ? '' : $headers['Authorization']);
		};

		$api_key = dol_string_nounprintableascii($api_key, 1);

		if (preg_match('/^dolcrypt:/i', $api_key)) {
			throw new RestException(503, 'Bad value for the API key. An API key should not start with dolcrypt:');
		}

		if ($api_key) {
			$userentity = 0;
			$token_rowid = 0;

			if (!getDolGlobalString('API_IN_TOKEN_TABLE')) {
				$sql = "SELECT u.login, u.datec, u.api_key as use_api, u.entity, u.api_key as api_key, u.entity as token_entity, 0 as token_rowid,";
				$sql .= " u.tms as date_modification";
				$sql .= " FROM ".MAIN_DB_PREFIX."user as u";
				$sql .= " WHERE u.api_key = '".$this->db->escape($api_key)."' OR u.api_key = '".$this->db->escape(dolEncrypt($api_key, '', '', 'dolibarr'))."'";
			} else {
				$sql = "SELECT u.login, u.datec, u.api_key as use_api, u.entity, oat.tokenstring as api_key, oat.entity as token_entity, rowid as token_rowid,";
				$sql .= " oat.tms as date_modification";
				$sql .= " FROM ".MAIN_DB_PREFIX."oauth_token AS oat";
				$sql .= " JOIN ".MAIN_DB_PREFIX."user AS u ON u.rowid = oat.fk_user";
				$sql .= " WHERE (oat.tokenstring = '".$this->db->escape($api_key)."'";
				$sql .= " OR oat.tokenstring = '".$this->db->escape(dolEncrypt($api_key, '', '', 'dolibarr'))."')";
				$sql .= " AND oat.service = 'dolibarr_rest_api'";
			}

			$result = $this->db->query($sql);
			if ($result) {
				$nbrows = $this->db->num_rows($result);
				if ($nbrows == 1) {
					$obj = $this->db->fetch_object($result);

					$login = $obj->login;
					$stored_key = dolDecrypt($obj->api_key);
					$userentity = $obj->entity;
					$token_entity = $obj->token_entity;
					$token_rowid = $obj->token_rowid;

					if (!defined("DOLENTITY") && $conf->entity != ($obj->entity ? $obj->entity : 1)) {		// If API was not forced with HTTP_DOLENTITY, and user is on another entity, so we reset entity to entity of user
						$conf->entity = ($obj->entity ? $obj->entity : 1);
						// We must also reload global conf to get params from the entity
						dol_syslog("Entity was not set on http header with HTTP_DOLAPIENTITY (recommended for performance purpose), so we switch now on entity of user (".$conf->entity.") and we have to reload configuration.", LOG_WARNING);
						$conf->setValues($this->db);

						// set global mysoc after setting conf entity (the entity can be changed with the user logged)
						// see master.inc.php
						require_once DOL_DOCUMENT_ROOT.'/societe/class/societe.class.php';

						$fmysoc = new Societe($this->db);
						$fmysoc->setMysoc($conf);

						// We set some specific default values according to country
						if ($fmysoc->country_code == 'DE' && !isset($conf->global->MAIN_INVERT_SENDER_RECIPIENT)) {
							// For DE, we need to invert our address with customer address
							$conf->global->MAIN_INVERT_SENDER_RECIPIENT = 1;
						}
						if ($fmysoc->country_code == 'FR' && !isset($conf->global->INVOICE_CATEGORY_OF_OPERATION)) {
							// For FR, default value of option to show category of operations is on by default. Decret n°2099-1299 2022-10-07
							$conf->global->INVOICE_CATEGORY_OF_OPERATION = 1;
						}
						if ($fmysoc->country_code == 'FR' && !isset($conf->global->INVOICE_DISABLE_REPLACEMENT)) {
							// For FR, the replacement invoice type is not allowed.
							// From an accounting point of view, this creates holes in the numbering of the invoice.
							// This is very problematic during a fiscal control.
							$conf->global->INVOICE_DISABLE_REPLACEMENT = 1;
						}
						if ($fmysoc->country_code == 'GR' && !isset($conf->global->INVOICE_DISABLE_REPLACEMENT)) {
							// The replacement invoice type is not allowed in Greece.
							$conf->global->INVOICE_DISABLE_REPLACEMENT = 1;
						}
						if ($fmysoc->country_code == 'GR' && !isset($conf->global->INVOICE_DISABLE_DEPOSIT)) {
							// The deposit invoice type is not allowed in Greece.
							$conf->global->INVOICE_DISABLE_DEPOSIT = 1;
						}

						if (($fmysoc->localtax1_assuj || $fmysoc->localtax2_assuj) && !isset($conf->global->MAIN_NO_INPUT_PRICE_WITH_TAX)) {
							// For countries using the 2nd or 3rd tax, we disable input/edit of lines using the price including tax (because 2nb and 3rd tax not yet taken into account).
							// Work In Progress to support all taxes into unit price entry when MAIN_UNIT_PRICE_WITH_TAX_IS_FOR_ALL_TAXES is set.
							$conf->global->MAIN_NO_INPUT_PRICE_WITH_TAX = 1;
						}
						// Set also the global variable $mysoc
						global $mysoc;
						$mysoc = $fmysoc;

						// Reload langs
						$langcode = getDolGlobalString('MAIN_LANG_DEFAULT', 'auto');
						if (!empty($user->conf->MAIN_LANG_DEFAULT)) {
							$langcode = $user->conf->MAIN_LANG_DEFAULT;
						}
						if ($langs->getDefaultLang() != $langcode) {
							$langs->setDefaultLang($langcode);
							$langs->tab_translate = array();
							$langs->loadLangs(array('main'));
						}
					}

					if ($conf->entity != ($token_entity ? $token_entity : 1)) {
						throw new RestException(401, "functions_isallowed::check_user_api_key Authentication KO for '".$login."': Token not valid (may be a typo or a wrong entity)");
					}
				} elseif ($nbrows > 1) {
					throw new RestException(503, 'Error when fetching user api_key : More than 1 user with this apikey');
				}
			} else {
				throw new RestException(503, 'Error when fetching user api_key :'.$this->db->error);
			}

			if ($login && $stored_key != $api_key) {		// This should not happen since we did a search on api_key
				$userClass::setCacheIdentifier($api_key);
				return false;
			}

			$genericmessageerroruser = 'Error user not valid (not found with api key or bad status or bad validity dates) (conf->entity='.$conf->entity.')';

			if (!$login) {
				dol_syslog("functions_isallowed::check_user_api_key Authentication KO for api key: Error when searching login user from api key", LOG_NOTICE);
				sleep(1); // Anti brute force protection. Must be same delay when user and password are not valid.
				throw new RestException(401, $genericmessageerroruser);
			}

			$fuser = new User($this->db);
			$result = $fuser->fetch(0, $login, '', 0, (empty($userentity) ? -1 : $conf->entity)); // If user is not entity 0, we search in working entity $conf->entity  (that may have been forced to a different value than user entity)
			if ($result <= 0) {
				dol_syslog("functions_isallowed::check_user_api_key Authentication KO for '".$login."': Failed to fetch on entity", LOG_NOTICE);
				sleep(1); // Anti brute force protection. Must be same delay when user and password are not valid.
				throw new RestException(401, $genericmessageerroruser);
			}

			// Check if user status is enabled
			if ($fuser->status != $fuser::STATUS_ENABLED) {
				// Status is disabled
				dol_syslog("functions_isallowed::check_user_api_key Authentication KO for '".$login."': The user has been disabled", LOG_NOTICE);
				sleep(1); // Anti brute force protection. Must be same delay when user and password are not valid.
				throw new RestException(401, $genericmessageerroruser);
			}

			// Check if session was unvalidated by a password change
			if (($fuser->flagdelsessionsbefore && !empty($_SESSION["dol_logindate"]) && $fuser->flagdelsessionsbefore > $_SESSION["dol_logindate"])) {
				// Session is no more valid
				dol_syslog("functions_isallowed::check_user_api_key Authentication KO for '".$login."': The user has a date for session invalidation = ".$fuser->flagdelsessionsbefore." and a session date = ".$_SESSION["dol_logindate"].". We must invalidate its sessions.");
				sleep(1); // Anti brute force protection. Must be same delay when user and password are not valid.
				throw new RestException(401, $genericmessageerroruser);
			}

			// Check date validity
			if ($fuser->isNotIntoValidityDateRange()) {
				// User validity dates are no more valid
				dol_syslog("functions_isallowed::check_user_api_key Authentication KO for '".$login."': The user login has a validity between [".$fuser->datestartvalidity." and ".$fuser->dateendvalidity."], current date is ".dol_now());
				sleep(1); // Anti brute force protection. Must be same delay when user and password are not valid.
				throw new RestException(401, $genericmessageerroruser);
			}

			// Increase counter of API access
			if (getDolGlobalString('API_COUNTER_ENABLED')) {
				if (!getDolGlobalString('API_IN_TOKEN_TABLE')) {
					// Update the counter into table llx_const
					include DOL_DOCUMENT_ROOT.'/core/lib/admin.lib.php';
					dolibarr_set_const($this->db, 'API_COUNTER_COUNT', getDolGlobalInt('API_COUNTER_COUNT') + 1);
					//var_dump('eeee');exit;
				} else {
					// Update the counter into table llx_oauth_token
					$tmpnow = dol_getdate(dol_now('gmt'), true, 'gmt');

					$sqlforcounter = "UPDATE ".$this->db->prefix()."oauth_token SET ";
					$sqlforcounter .= " apicount_total = apicount_total + 1,";
					$sqlforcounter .= " apicount_month = apicount_month + 1,";
					// if last access was done during previous month, we save pageview_month into pageviews_previous_month
					$sqlforcounter .= " pageviews_previous_month = ".$this->db->ifsql("lastaccess < '".$this->db->idate(dol_mktime(0, 0, 0, $tmpnow['mon'], 1, $tmpnow['year'], 'gmt', 0), 'gmt')."'", 'apicount_month', 'apicount_previous_month').",";
					$sqlforcounter .= " lastaccess = '".$this->db->idate(dol_now('gmt'), 'gmt')."'";
					$sqlforcounter .= " WHERE rowid = ".((int) $token_rowid);

					$this->db->query($sqlforcounter);
				}
			}

			// User seems valid
			$fuser->loadRights();

			// Set the property $user to the $user of API
			static::$user = $fuser;

			// Set also the global variable $user to the $user of API
			$user = $fuser;

			if ($fuser->socid) {
				static::$role = 'external';
			}

			if ($fuser->admin) {
				static::$role = 'admin';
			}
		} else {
			throw new RestException(401, "Failed to login to API. Neither parameter 'HTTP_DOLAPIKEY' nor 'Authentication: Bearer' found on HTTP header (and no parameter DOLAPIKEY in URL).");
		}

		$userClass::setCacheIdentifier(static::$role);
		Resources::$accessControlFunction = 'DolibarrApiAccess::verifyAccess';
		$requirefortest = static::$requires;
		if (!is_array($requirefortest)) {
			$requirefortest = explode(',', $requirefortest);
		}
		return in_array(static::$role, (array) $requirefortest) || static::$role == 'admin';
	}

	// phpcs:disable PEAR.NamingConventions.ValidFunctionName
	/**
	 * @return string string to be used with WWW-Authenticate header
	 */
	public function __getWWWAuthenticateString()
	{
		// phpcs:enable
		return '';
	}

	/**
	 * Verify access
	 *
	 * @param   array{class:array{DolibarrApiAccess:array{properties:array{requires?:bool}}}} $m Properties of method
	 *
	 * @access private
	 * @return bool
	 */
	public static function verifyAccess(array $m)
	{
		$requires = isset($m['class']['DolibarrApiAccess']['properties']['requires'])
				? $m['class']['DolibarrApiAccess']['properties']['requires']
				: false;


		return $requires
			? static::$role == 'admin' || in_array(static::$role, (array) $requires)
			: true;
	}
}
